{*************************************************************}
{            RegControls for Delphi 32                        }
{ Version:   1.2                                              }
{ Author:    Aleksey Kuznetsov                                }
{ E-Mail:    info@utilmind.com                                }
{ Home Page: www.utilmind.com                                 }
{ Created:   May, 16, 1999                                    }
{ Modified:  September, 2, 1999                               }
{ Legal:     Copyright (c) 1999, UtilMind Solutions           }
{*************************************************************}
{ TRegForm - automaticly saves form placement and size to     }
{            SystemRegistry                                   }
{ TRegComboBox - saves history list to SystemRegistry.        }
{                When user type in a few chars, if that value }
{                is already in the combobox it will fill in.  }
{*************************************************************}
{ PROPERTIES:                                                 }
{   RegEnabled: Boolean - Use Registry or not                 }
{   RegLocation: TRegLocation - Current_User or Local_Machine }
{   RegKey: String - Key in System Registry for storing data. }
{                    Be extremely cautious by setting or      }
{                    changing this property                   }
{-------------------------------------------------------------}
{ New properties in v1.1 (for TRegComboBox only)              }
{   RegAutoFindValue: Boolean - if True, when user type in a  }
{                            few chars, if that value is      }
{                            already in the combobox it will  }
{                            fill in.                         }
{   RegAutoFindDelay: Integer - delay before substitution of  }
{                               pre-entered posible value.    }
{   RegAutoFindCaseSensitive: Boolean - ignore case during    }
{                                       finding value.        }
{   RegSaveOnExit: Boolean - if True then value will saved    }
{                            automaticaly OnExit.             }
{-------------------------------------------------------------}
{ METHODS (for TRegComboBox only):                            }
{   Save - added current. Text to history and saves it to     }
{          SystemRegistry.                                    }
{*************************************************************}
{   Updates in v1.2                                           }
{                                                             }
{ 1. Removed all static variables to make the controls        }
{    multi-thread safe.                                       }
{                                                             }
{ 2. Fixed save function which deleted the selected text.     }
{    When deleting items, you should loop from top downto     }
{    bottom, not from 0 to top.                               }
{                                                             }
{ 3. Fixed casesensitive by adding NOT. When you set          }
{    sensitive to false, it should lowercase both args, you   }
{    were lowercasing when true.                              }
{                                                             }
{ 4. Changed search to select the smallest matching item, not }
{    the first matching item. Think, this is better for any   }
{    applications.                                            }
{                                                             }
{ 5. Moved Creation of Reg outside 'try'. If create fails,    }
{    you do not need to call the destructor.                  }
{                                                             }
{ 6. Prevent storing empty key names in registry or list if   }
{    value is blank.                                          }
{                                                             }
{ 7. Suppress autofind immediately after a backspace.         }
{                                                             }
{*************************************************************}
{ Please see demo program for more information.               }
{*************************************************************}

unit RegControls;

interface

uses
  Windows, Messages, Classes, Controls, StdCtrls, ExtCtrls,
  Forms, Registry, SysUtils;

type
  TRegLocation = (CurrentUser, LocalMachine);

{ ----- TRegForm ----- }

  TRegForm = class(TComponent)
  private
    ParentForm: TForm;
    OldCloseQuery: TCloseQueryEvent;
    FRegEnabled: Boolean;
    FRegLocation: TRegLocation;
    FRegKey: String;

    procedure SetRegLocation(Value: TRegLocation);
    procedure CloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure DoSave(Reg: TRegistry);
    procedure SetRegKey(Value: String);
  protected
    procedure Loaded; override;
  public
    constructor Create(aOwner: TComponent); override;
    procedure Save;
  published
    property RegEnabled: Boolean read FRegEnabled write FRegEnabled;
    property RegLocation: TRegLocation read FRegLocation write SetRegLocation;
    property RegKey: String read FRegKey write SetRegKey;
  end;

{ ----- TRegComboBox ----- }

  TRegComboBox = class(TComboBox)
  private
    FRegAutoFindValue: Boolean;
    FRegAutoFindDelay: Integer;
    FRegAutoFindCaseSensitive: Boolean;
    FRegEnabled: Boolean;
    FRegLocation: TRegLocation;
    FRegKey: String;
    FMaxCount: Integer;
    FRegSaveOnExit: Boolean;

    Timer: TTimer;

    procedure SetRegAutoFindDelay(Value: Integer);
    procedure SetRegLocation(Value: TRegLocation);
    procedure DoSave(Reg: TRegistry);
    procedure SetRegKey(Value: String);
    procedure OnTimer(Sender: TObject);
    procedure CNChar(var Msg: TWMChar); message CN_CHAR;
    procedure WMKillFocus(var Msg: TMessage); message wm_KillFocus;
  protected
    procedure Loaded; override;
  public
    constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;
    procedure Save;
  published
    property RegAutoFindValue: Boolean read FRegAutoFindValue write FRegAutoFindValue;
    property RegAutoFindDelay: Integer read FRegAutoFindDelay write SetRegAutoFindDelay;
    property RegAutoFindCaseSensitive: Boolean read FRegAutoFindCaseSensitive write FRegAutoFindCaseSensitive;
    property RegEnabled: Boolean read FRegEnabled write FRegEnabled;
    property RegLocation: TRegLocation read FRegLocation write SetRegLocation;
    property RegKey: String read FRegKey write SetRegKey;
    property RegMaxCount: Integer read FMaxCount write FMaxCount;
    property RegSaveOnExit: Boolean read FRegSaveOnExit write FRegSaveOnExit;
  end;

procedure Register;

implementation

{ ----- TRegForm ----- }

constructor TRegForm.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);
  ParentForm := TForm(aOwner);
  OldCloseQuery := ParentForm.OnCloseQuery;
  ParentForm.OnCloseQuery := CloseQuery;
  FRegEnabled := True;
  FRegKey := '\Software\Default';
end;

procedure TRegForm.Loaded;
var
  Reg           : TRegistry;
  WS            : TWindowState;
begin
  inherited Loaded;
  if FRegEnabled and not (csDesigning in ComponentState) then
    begin
      Reg := TRegistry.Create;
      try
        try
          if FRegLocation = CurrentUser then
            Reg.RootKey := HKEY_CURRENT_USER
          else
            Reg.RootKey := HKEY_LOCAL_MACHINE;
          Reg.OpenKey(FRegKey, False);
          Reg.ReadBinaryData('RCWindowState', WS, SizeOf(WS));
          ParentForm.WindowState := WS;
          if WS <> wsMaximized then
            begin
              ParentForm.Left := Reg.ReadInteger('RCLeft');
              ParentForm.Top := Reg.ReadInteger('RCTop');
              ParentForm.Width := Reg.ReadInteger('RCWidth');
              ParentForm.Height := Reg.ReadInteger('RCHeight');
            end;
        except
        end;
      finally
        Reg.Free;
      end;
    end;
end;

procedure TRegForm.DoSave(Reg: Tregistry);
var
  WS            : TWindowState;
begin
  if ParentForm <> nil then
    with Reg do
      begin
        WriteInteger('RCLeft', ParentForm.Left);
        WriteInteger('RCTop', ParentForm.Top);
        WriteInteger('RCWidth', ParentForm.Width);
        WriteInteger('RCHeight', ParentForm.Height);
        WS := ParentForm.WindowState;
        WriteBinaryData('RCWindowState', WS, SizeOf(WS));
      end;
end;

procedure TRegForm.Save;
var
  Reg           : TRegistry;
begin
  if not (csDesigning in ComponentState) then
    begin
      Reg := TRegistry.Create;
      try
        try
          if FRegLocation = CurrentUser then
            Reg.RootKey := HKEY_CURRENT_USER
          else
            Reg.RootKey := HKEY_LOCAL_MACHINE;
          Reg.OpenKey(FRegKey, True);
          DoSave(Reg);
        except
        end;
      finally
        Reg.Free;
      end;
    end;
end;

procedure TRegForm.CloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  if Assigned(OldCloseQuery) then
    OldCloseQuery(Sender, CanClose);
  if FRegEnabled and CanClose and (ParentForm.Handle <> 0) then
    try
      Save;
    except
      Application.HandleException(Self);
    end;
end;

procedure TRegForm.SetRegKey(Value: string);
begin
  if (Value <> '') and (Value <> FRegKey) then
    begin
      if Value[Length(Value)] = '\' then
        SetLength(Value, Length(Value) - 1);
      FRegKey := Value;
    end;
end;

procedure TRegForm.SetRegLocation(Value: TRegLocation);
var
  Reg           : TRegistry;
begin
  if FRegLocation <> Value then
    begin
      FRegLocation := Value;
      if not (csDesigning in ComponentState) and not (csReading in ComponentState) and
        not (csLoading in ComponentState) then
        begin
          Reg := TRegistry.Create;
          try
            try
              Reg.RootKey := HKEY_CURRENT_USER;
              try
                Reg.DeleteKey(FRegKey);
              except
              end;
              Reg.RootKey := HKEY_LOCAL_MACHINE;
              try
                Reg.DeleteKey(FRegKey);
              except
              end;
              if FRegLocation = CurrentUser then
                Reg.RootKey := HKEY_CURRENT_USER
              else
                Reg.RootKey := HKEY_LOCAL_MACHINE;
              Reg.OpenKey(FRegKey, True);
              try
                DoSave(Reg);
              except
              end;
            except
            end;
          finally
            Reg.Free;
          end;
        end;
    end;
end;

{ ----- TRegComboBox ----- }

constructor TRegComboBox.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);
  FRegAutoFindValue := True;
  FRegAutoFindDelay := 500;
  FRegEnabled := True;
  FMaxCount := 20;
  FRegKey := '\Software\Default';
  Timer := TTimer.Create(Self);
  Timer.Enabled := False;
  Timer.Interval := FRegAutoFindDelay;
  Timer.OnTimer := OnTimer;
end;

destructor TRegComboBox.Destroy;
begin
  Timer.Destroy;
  inherited Destroy;
end;

procedure TRegComboBox.Loaded;
var
  Reg           : TRegistry;
  St            : TStringList;
begin
  inherited Loaded;
  if FRegEnabled and not (csDesigning in ComponentState) then
    begin
      Reg := TRegistry.Create;
      try
        try
          if FRegLocation = CurrentUser then
            Reg.RootKey := HKEY_CURRENT_USER
          else
            Reg.RootKey := HKEY_LOCAL_MACHINE;
          Reg.OpenKey(FRegKey, False);
          St := TStringList.Create;
          Reg.GetValueNames(St);
          Items.Assign(St);
          St.Destroy;
        except
        end;
      finally
        Reg.Free;
      end;
    end;
end;

procedure TRegComboBox.DoSave(Reg: Tregistry);
var
  i             : Integer;
begin
  Reg.DeleteKey(FRegKey);
  Reg.OpenKey(FRegKey, True);
  for i := 0 to pred(items.count) do
    begin
      if (i >= FMaxCount) then exit;
      if (trim(items[i]) <> '') then Reg.WriteString(Items[i], '');
    end;
  Reg.Closekey;
end;

procedure TRegComboBox.Save;
var
  i, c          : Integer;
  s             : String;
  Reg           : TRegistry;
begin
  if not (csDesigning in ComponentState) and (Text <> '') then
    begin
      s := Text;
      c := Items.Count;
      for i := Pred(c) downto 0 do
        if (Items[i] = s) or (items[i] = '') then
          begin
            Items.Delete(i);
          end;
      Items.Insert(0, s);
      ItemIndex := 0;
      Reg := TRegistry.Create;
      try
        try
          if FRegLocation = CurrentUser then
            Reg.RootKey := HKEY_CURRENT_USER
          else
            Reg.RootKey := HKEY_LOCAL_MACHINE;
          DoSave(Reg);
        except
        end;
      finally
        Reg.Free;
      end;
    end;
end;

procedure TRegComboBox.SetRegKey(Value: string);
begin
  if (Value <> '') and (Value <> FRegKey) then
    begin
      if Value[Length(Value)] = '\' then
        SetLength(Value, Length(Value) - 1);
      FRegKey := Value;
    end;
end;

procedure TRegComboBox.SetRegAutoFindDelay(Value: Integer);
begin
  if FRegAutoFindDelay <> Value then
    begin
      FRegAutoFindDelay := Value;
      Timer.Interval := FRegAutoFindDelay;
    end;
end;

procedure TRegComboBox.SetRegLocation(Value: TRegLocation);
var
  Reg           : TRegistry;
begin
  if FRegLocation <> Value then
    begin
      FRegLocation := Value;
      if not (csDesigning in ComponentState) and not (csReading in ComponentState) and
        not (csLoading in ComponentState) then
        begin
          Reg := TRegistry.Create;
          try
            try
              Reg.RootKey := HKEY_CURRENT_USER;
              try
                Reg.DeleteKey(FRegKey);
              except
              end;
              Reg.RootKey := HKEY_LOCAL_MACHINE;
              try
                Reg.DeleteKey(FRegKey);
              except
              end;
              if FRegLocation = CurrentUser then
                Reg.RootKey := HKEY_CURRENT_USER
              else
                Reg.RootKey := HKEY_LOCAL_MACHINE;
              Reg.OpenKey(FRegKey, True);
              try
                DoSave(Reg);
              except
              end;
            except
            end;
          finally
            Reg.Free;
          end;
        end;
    end;
end;

procedure TRegComboBox.OnTimer;
var
  i, preLength  : Integer;
  l             : Integer;
  c             : Integer;
  comptext      : String;
  testtext      : String;
  foundx        : Integer;      // index of found item
  foundl        : Integer;      // length of found item
begin
  if not FRegAutoFindValue then Exit;
  if Text = '' then Exit;
  c := Items.Count;
  if c = 0 then Exit;
  foundx := -1;
  foundl := 999;
  preLength := Length(Text);
  if FRegAutoFindCaseSensitive then
    comptext := text
  else
    comptext := lowercase(text);
  for i := 0 to pred(c) do
    begin
      l := length(Items[i]);
      testtext := Copy(Items[i], 1, Length(Text));
      if not FRegAutoFindCaseSensitive then
        testtext := LowerCase(testtext);
      if testtext = comptext then
        begin
          if l < foundl then
            begin
              foundl := l;
              foundx := i;
            end;
        end;
    end;

  if (foundx <> -1) then
    begin
      if FRegAutoFindCaseSensitive then
        Text := Items[foundx]
      else
        Text := Text + Copy(Items[foundx], preLength + 1, Length(Items[foundx]) - preLength);
      Timer.Enabled := False;
      if not Timer.Enabled then
        begin
          SendMessage(Handle, CB_SETEDITSEL, 0, MakeLParam(preLength, Length(Text)));
        end;
    end;
end;

procedure TRegComboBox.CNChar(var Msg: TWMChar);
begin
  inherited;
  if FRegAutoFindValue and
    (FRegAutoFindDelay > 0) then
    begin
      if (Msg.charcode <> VK_BACK) then
        Timer.Enabled := True
      else
        Timer.Enabled := False;
    end
  else
    OnTimer(nil);
  Msg.result := 0;
end;

procedure TRegComboBox.WMKillFocus(var Msg: TMessage);
begin
  inherited;
  if FRegSaveOnExit then Save;
end;

procedure Register;
begin
  RegisterComponents('UtilMind', [TRegForm, TRegComboBox]);
end;

end.

